#include <iostream>
#include <getopt.h>

#include "param.hpp"
#include "io.hpp"
#include "solver.hpp"
#include "tools.hpp"

using namespace std;

void usage() {
	cout << "Usage: ./sim.exe [--output output-file] [--seed seed-value] < input-file" << endl;
	exit(-1);
}

int main(int argc, char *argv[]) {
	
	char *out_filename = NULL;
	uint32_t seed_v = 214;
	
	// Get options
	struct option longopts[] = {
		{"output", required_argument, NULL, 'o'},
		{"seed", required_argument, NULL, 's'},
		{0, 0, 0, 0}
	};
	int opt, optidx;
	while((opt = getopt_long(argc, argv, "o:s:", longopts, &optidx)) != -1) {
		switch(opt) {
		case 'o':
			out_filename = optarg;
			break;
		case 's':
			sscanf(optarg, "%u", &seed_v);
			break;
		default:
			usage();
		}
	}
	
	// Init seed value
	uint32_t seed[3];
	srand(seed_v);
	seed[0] = rand();
	seed[1] = rand();
	seed[2] = rand();
	
	// W: Max width of a board (specified in a problem file)
	// H: Max height of a board (specified in a problem file)
	// blocks: # of blocks (specified in a problem file)
	short int W, H, blocks, line_num;
	// W_ext: Width of a current solution
	// H_ext: Height of a current solution
	short int W_ext, H_ext;
	
	// block data
	short int block_info[MAX_BLOCKS+1][5][3];
	
	// line data
	short int line_info[MAX_LINES+1][2][5];
	
	// block positions on slot matrix
	short int block_place_global[MAX_BLOCKS+1][2];

	// An opt. result will be stored in the array
	short int opt_result[MAX_CELLS];
	
	// Check problem
	read_problem(&W, &H, &blocks, &line_num, block_info, line_info);
	extract_line_info(line_num, blocks, block_info, line_info);

	// Solver body
	global_placement(seed, W, H, line_num, blocks, block_info, line_info, block_place_global);

	hls::stream<ap_uint<4> > state;
	bool solved = solver(state, seed, &blocks, &line_num, block_info, &W_ext, &H_ext, block_place_global, opt_result);
	
	// Check answer
	if(!solved) {
		cout << "Fail re-routing" << endl;
	}
	else {
		cout << "== Answer ==" << endl;
		cout << "SIZE " << W_ext << "X" << H_ext << endl;
		show_result(line_num, blocks, W_ext, H_ext, opt_result);
	}
	if(W_ext > W || H_ext > H) {
		cout << "Fail satisfying constraint T_T" << endl;
	}
	else {
		cout << "Satisfy constraint ^_^" << endl;
		if(out_filename == NULL) return 0;
		cout << "Output to " << out_filename << endl;
		short int block_place_basis[MAX_BLOCKS+1][2];
		for(int i = 1; i <= blocks; i++) {
			block_place_basis[i][0] = block_info[i][0][0];
			block_place_basis[i][1] = block_info[i][0][1];
		}
		output_to_file(out_filename, line_num, blocks, W_ext, H_ext, opt_result, block_place_basis);
	}
	
	return 0;
}
